
#ifndef _BC5602_H_
#define _BC5602_H_

#include		"ht8.h"
#include 	"bc5602reg.h"

/*-------MACRO----------------------------------------------------------------*/
#define	SET_BIT(pos) 	((u8)(1<<((u8)(pos))))
#define	B2W(hb,lb)		((u16)((hb << 8)+lb))
/*----------------- Configuration Control Register -----------------*/
enum
{
   REGS_BANK0 = 0,        			/* register bank 0 */
   REGS_BANK1,                	/* register bank 1 */
   REGS_BANK2,                	/* register bank 2 */
   REGS_BANK3                 	/* register bank 3 */
};

enum
{
	N03DBM=0,							/* TX power -3dBm */
	P00DBM,								/* TX power 0dBm */
	P05DBM,								/* TX power +5dBm */
	P07DBM								/* TX power +7dBm */
};

enum
{
	ARD_DLY250=0,
	ARD_DLY500,
	ARD_DLY750,
	ARD_DLY1000,
	ARD_DLY1250,
	ARD_DLY1500,
	ARD_DLY1750,
	ARD_DLY2000,
	ARD_DLY2250,
	ARD_DLY2500,
	ARD_DLY2750,
	ARD_DLY3000,
	ARD_DLY3250,
	ARD_DLY3500,
	ARD_DLY3750,
	ARD_DLY4000	
};

#define	_DIR_EN_						0x10	/* Direct Mode enable */
#define	_AGC_EN_						0x40  /* AGC enable */

typedef union 
{
   unsigned char	value;
   struct 
   {
      unsigned char  bank : 2;
		const unsigned char : 2;		
      unsigned char  dir_en : 1;
		const unsigned char : 1;		
      unsigned char  agc_en : 1;
      const unsigned char : 1;
   }bits;
}config_t;


/*----------------- Reset/Clock Control Register 1 -----------------*/
#define	_RST_LL_						0x01	/* reset low voltage (1.2V) logic */
#define	_FCLKO_EN_              0x02	/* clock output enable */
#define	_PWRON_						0x80	

enum
{
   FCLKO_DIV1 = 0,       		/* Clock output dividend from XCLK */
   FCLKO_DIV2,
   FCLKO_DIV4,
   FCLKO_DIV8,
};

#define	_XCLK_EN_					0x10	/* XCLK enable. Enable XCLK path to baseband block. Set 0 to save power. */
#define	_XCLK_RDY_					0x20	/* XCLK ready flag. */
#define	_FSYCK_RDY_					0x40	/* FSYCK ready flag. */
#define	_PWRON_						0x80	/* 3.3V power on flag to indicate the first time power on. */

typedef union 
{
   unsigned char	value;
   struct 
   {
      unsigned char  rstll : 1;
      unsigned char  fsyck_en : 1;
      unsigned char  fsyck_div : 2;
      unsigned char  xclk_en : 1;
      unsigned char  xclk_rdy : 1;
      unsigned char  fsyck_rdy : 1;
      unsigned char  pwron : 1;
   }bits;
}clk_ctl_t;

/*----------------- Reset/Clock Control Register 2 -----------------*/
#define	_ADCLK_SEL_					0x01
typedef union 
{
   unsigned char	value;
   struct 
   {
      unsigned char  adc_clk : 1;
      const unsigned char : 7;
   }bits;
}adclk_ctl_t;

/*-----------------  Interrupt Control Register 1 -----------------*/
#define	_PRIM_RX_					0x01	/* RX/TX control 0:PTX ,1:PRX */
#define	_MASK_MAXRT_				0x10
#define	_MASK_TXDS_					0x20
#define	_MASK_RXDR_					0x40

enum
{
	PRIM_PTX = 0,            	/**< Primary TX operation */
	PRIM_PRX             		/**< Primary RX operation */
};

enum
{
	IRQ_MAXRT = 4,     			/**< Max retries interrupt */
	IRQ_TXDS,          			/**< TX data sent interrupt */
	IRQ_RXDR,           			/**< RX data received interrupt */
	IRQ_ALL = 0xFF
};

typedef union 
{
   unsigned char	value;
   struct 
   {
      unsigned char  prim_rx : 1;
      const unsigned char : 3;		
      unsigned char  mask_maxrt : 1;
      unsigned char  mask_txds : 1;
      unsigned char  mask_rxdr : 1;
      const unsigned char : 1;
   }bits;
}prim_ctl_t;

/*-----------------  Interrupt Control Register 2/3 -----------------*/
#define	_TX_FULL_					0x01	/* TX FIFO full flag 1: TX full,0:Available locations in TX FIFO*/
enum
{
	RX_PIPE0 = 0,								/* Data pipe number for the payload available for reading from RX_FIFO */
	RX_PIPE1,
	RX_PIPE2,
	RX_PIPE3,
	RX_PIPE4,
	RX_PIPE5,
	RX_EMPTY = 7								/* RX FIFO Empty */
};
#define	_MAX_RT_         			0x10  /* Maximum number of TX retransmits interrupt */
#define	_TX_DS_      				0x20  /* Data Sent TX FIFO interrupt */
#define	_RX_DR_         			0x40  /* Data Ready RX FIFO interrupt */

typedef union 
{
   unsigned char	value;
   struct 
   {
	   unsigned char tx_full : 1;
		unsigned char rx_p_no : 3;
		unsigned char max_rt : 1;
		unsigned char tx_ds : 1;
		unsigned char rx_dr : 1;
      const unsigned char : 1;		
   }bits;
}irq_status_t;

/*-----------------  FIFO status Register -----------------*/
#define	_RX_FF_EMPTY_				0x01
#define	_RX_FF_FULL_				0x02
#define	_TX_FF_EMPTY_				0x10
#define	_TX_FF_FULL_				0x20

typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char rx_empty : 1;
	   unsigned char rx_full : 1;
		const unsigned char : 2;
	   unsigned char tx_empty : 1;
	   unsigned char tx_full : 1;
		const unsigned char : 2;
	} bits;
} fifo_status_t;

/*------------------ IO Control Register 1 -------------------------*/
enum 
{
   GPIO1 = 0,
   GPIO2,
   GPIO3,
   GPIO4
};

enum 
{
   INPUT_MODE = 0,         /* GPIO input mode */
   SPI_SDO_FUN,            /* SPI SDO function */
   DIRECT_TRXD,            /* Direct mode TX/RD data input/output */
   DIRECT_TXD,             /* Direct mode TX data input */
   DIRECT_RXD,             /* Direct mode RX data output */
   INT_REQUEST,            /* Interrupt request output */
   LOSC_INPUT,             /* LOSC input */
   TBCLK_OUTPUT = 8,       /* Direct mode TX bit clock output */
   RBCLK_OUTPUT,           /* Direct mode RX bit clock output */
   FCLK_OUTPUT,            /* Frequency clock output */
   LOSC_OUTPUT,            /* LOSC output */
   EPA_CTLO,               /* External PA control output */
   ELNA_CTLO,              /* External LNA control output */
   DIRECT_TRBCLK           /* Direct mode Tx/Rx bit clock output */
};

enum 
{
	GDCS500uA = 0,				/* GPIO current strength 0.5mA */
	GDCS1mA,                /* GPIO current strength 1mA */
	GDCS5mA,                /* GPIO current strength 5mA */
	GDCS10mA                /* GPIO current strength 10mA */
};

typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char gpio1s : 3;
	   unsigned char gpio2s : 3;
	   unsigned char padds : 2;
	} bits;
} gpio12_ctl_t;

typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char gpio3s : 4;
	   unsigned char gpio4s : 4;
	} bits;
} gpio34_ctl_t;

/*------------------ IO Control Register 3 -------------------------*/
#define 	_GPIOPU1_					0x02	/* Pull high enable */
#define 	_GPIOPU2_					0x04	/* Pull high enable */
#define 	_GPIOPU3_					0x08	/* Pull high enable */
#define 	_GPIOPU4_					0x10	/* Pull high enable */
#define 	_SPIPU_						0x40	/* 3 wire SPI pull up enable. */
#define 	_SDOTEN_						0x80	/* SDO tri-state enable */


#define  AW_3BYTE   1
#define	 AW_4BYTE   2
#define  AW_5BYTE   3


typedef union 
{
   unsigned char value;
	struct 
   {
		const unsigned char : 1;		
	   unsigned char gpio1pu : 1;
	   unsigned char gpio2pu : 1;
	   unsigned char gpio3pu : 1;
	   unsigned char gpio4pu : 1;
		const unsigned char : 1;		
	   unsigned char spipu : 1;
	   unsigned char sdoten : 1;		
	} bits;
} gpiopu_ctl_t;

/*------------------ Packet Control Register 1 -------------------------*/
#define	_CRC8_						0x10	/* CRC encoding scheme */
#define	_CRC_EN_						0x20	/* CRC field enable */
enum
{
	CRC_OFF = 0,
	CRC_8BIT,
	CRC_16BIT
};

typedef union 
{
   unsigned char value;
	struct 
   {
		const unsigned char : 4;		
	   unsigned char crc8 : 1;
	   unsigned char crc_en : 1;		
		const unsigned char : 2;		
	} bits;
} crc_ctl_t;

/*------------------ Packet Control Register 2 -------------------------*/
#define 	_WHT_EN_						0x80	/* Data Whitening enable. */
typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char whtsd : 7;
	   unsigned char wht_en : 1;
	} bits;
} whitening_ctl_t;

/*------------------ Packet Control Register 3 -------------------------*/
#define 	_WHT_PCF_EN_				0x80	/* Data Whitening range enable. */
typedef union 
{
   unsigned char value;
	struct 
   {
		const unsigned char : 7;		
	   unsigned char pcf_en : 1;
	} bits;
} wht_pcf_ctl_t;

/*------------------ RF channel Control Register -------------------------*/
typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char rf_ch : 7;
		const unsigned char : 1;
	} bits;
} rf_ch_t;
/*------------------ De-modulator Control Register 1 -------------------------*/
enum
{
	DR500KBPS = 0,
	DR250KBPS,
	DR125KBPS,
};

typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char dr : 3;
		const unsigned char : 3;		
	   unsigned char aw : 2;
	} bits;
} setup_aw_t;
/*------------------ PTX Retransmission Control Register 1 -------------------------*/
typedef union 
{
   unsigned char value;
	struct 
   {
		unsigned char arc : 4;
		unsigned char ard : 4;
	} bits;
} setup_retr_t;

/*------------------ PTX Retransmission Control Register 2 -------------------------*/
typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char arc_cnt : 4;
		unsigned char plos_cnt : 4;
	} bits;
} observe_tx_t;

/*------------------ Chip Enable Control Register -------------------------*/
#define	_CE_							0x01	
typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char ce : 1;
		const unsigned char : 7;
	} bits;
} chip_ctl_t;

/*------------------ Operation Mode Control Register -------------------------*/
#define _SX_EN_						0x01	/* Synthesizer Enable (standby mode enable) */
#define _RTX_SEL_					0x02	/* RX and TX mode selection: 0: Rx mode, 1:TX mode */
#define _RTX_EN_					0x04	/* RX or TX mode enable, selected by RTX_SEL */
#define _ACAL_EN_					0x08	/* Auto calibration enable. */
#define _PWR_SOFT_					0x80	/* RF normal operation mode, */

typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char sx_en : 1;
	   unsigned char rtx_sel : 1;
	   unsigned char rtx_en : 1;
	   unsigned char acal_en : 1;		
		const unsigned char : 3;
	   unsigned char pwr_soft : 1;		
	} bits;
} om_ctl_t;

/*------------------ CFO Control Register 1 -------------------------*/
#define	_AMBLE2_						0x40
#define	_AFCO_EN_					0x80
typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char mask_b0 : 4;
		const unsigned char : 2;
	   unsigned char amble2 : 1;
	   unsigned char afco_en : 1;		
	} bits;
} cfo_ctl_t;

/*------------------ Operation Mode Status Register -------------------------*/
#define	_CD_FLAG_					0x10

enum
{
	_DEEP_SLEEP_MODE_ = 0,
	_MIDDLE_SLEEP_MODE_,
	_LIGHT_SLEEP_MODE_,
	_STANDBY_MODE_,
	_TX_MODE_,
	_RX_MODE_,
	_CALIBRATION_MODE_,
};	

typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char omst : 3;
		const unsigned char : 1;
	   unsigned char cd_flag : 1;
		const unsigned char : 3;
	} bits;
} om_status_t;

/*------------------ RSSI Control Register 1 -------------------------*/
typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char cthd : 4;
		const unsigned char : 4;
	} bits;
} rssi_cthd_t;


/**
 * Typedef for the EN_AA, EN_RXADDR and DYNPD registers. Contains all the 
 * bitaddressable settings in the bits struct and the value sent to the radio 
 * in the UINT8 
 */

enum
{
	TRXADDR = 0,             			
	PIPE0 = 0,							/**< Select pipe0 */
	PIPE1,              				/**< Select pipe1 */
	PIPE2,              				/**< Select pipe2 */
	PIPE3,              				/**< Select pipe3 */
	PIPE4,              				/**< Select pipe4 */
	PIPE5,              				/**< Select pipe5 */
   PIPEALL = 0xFF          		/**< Close or open all pipes*/
};

typedef union 
{
   unsigned char value;
   struct 
   {
		unsigned char pipe0 : 1;
		unsigned char pipe1 : 1;
		unsigned char pipe2 : 1;
		unsigned char pipe3 : 1;
		unsigned char pipe4 : 1;
		unsigned char pipe5 : 1;
		const unsigned char : 2;
   }bits;
} en_pipes_t;

/**
 * Typedef for the FEATURE register. Contains all the bitaddressable 
 * settings in the bits struct and the value sent to the radio in the U8
 */
#define	_EN_DYN_ACK_		0x01
#define	_EN_ACK_PAY_		0x02
#define	_EN_DYN_DPL_		0x04
typedef union 
{
   unsigned char value;
	struct 
   {
		unsigned char en_dyn_ack : 1;
		unsigned char en_ack_pay : 1;
		unsigned char en_dpl : 1;
		const unsigned char : 5;
	} bits;
} feature_t;

/**
 * Typedef for the RX_PW_Px registers. Contains all the bitaddressable 
 * settings in the bits struct and the value sent to the radio in the U8
 */
typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char rx_pw : 6;
		const unsigned char : 2;
	} bits;
} rx_pw_t;

/*------------------ XO Control Register 1 -------------------------*/
#define _XO_IL_						0x20	/* crystal oscillator low current mode. */
typedef union 
{
   unsigned char value;
	struct 
   {
	   unsigned char xo_trim : 5;
	   unsigned char xo_il : 1;
	   unsigned char : 2;
	} bits;
} xo_cap_t;
/*------------------ XO Control Register 2 -------------------------*/
#define 	_XODIV2_						0x08	/* Select Div-by-2 output for XO */
#define 	_XO_SEL_						0x10	/* Select crystal */
#define	_ADCLKDIV2_					0x80

#define	XCLK32M						0x00
#define	XCLK16M						0x10

typedef union 
{
   unsigned char value;
	struct 
   {
		const unsigned char : 3;		
	   unsigned char xodiv2 : 1;		
	   unsigned char xo_sel : 1;
		const unsigned char : 2;
	   unsigned char adclkdiv2 : 1;
	} bits;
} xclk_ctl_t;

void  BC5602InterfaceConfigure(void);
void  BC5602GPIOConfigure(u8 gpio,u8 fun);

void  BC5602StrobeCommand(u8 cmd);	
u8    BC5602ReadRegister(u8 regs);
void  BC5602WriteRegister(u8 regs,u8 data);
void  BC5602ReadMultibyteRegister(u8 cmd,u8 *pbuf,u8 length);
void  BC5602WriteMultibyteRegister(u8 cmd,u8 *pbuf,u8 length);
void 	BC5602WriteConstRegister(u8 length,uc16 *pbuf);

#define  BC5602SoftwareReset()			BC5602StrobeCommand(SOFT_RESET_CMD)
#define  BC5602RegisterBank(regbk)   	BC5602StrobeCommand(REGS_BANK_CMD+(regbk & 0x03))
#define  BC5602DeepSleepMode()      	BC5602StrobeCommand(DEEP_SLEEP_CMD)
#define  BC5602MiddleSleepMode()    	BC5602StrobeCommand(MIDDLE_SLEEP_CMD)
#define  BC5602LightSleepMode()			BC5602StrobeCommand(LIGHT_SLEEP_CMD)
#define  BC5602StandbyMode()				BC5602StrobeCommand(STANDBY_MODE_CMD)
#define  BC5602TransmitterMode()			BC5602StrobeCommand(TX_MODE_CMD)
#define  BC5602ReceiveMode()				BC5602StrobeCommand(RX_MODE_CMD)
#define  BC5602TxFIFOFlush()				BC5602StrobeCommand(TX_FIFO_FLUSH_CMD)
#define  BC5602RxFIFOFlush()				BC5602StrobeCommand(RX_FIFO_FLUSH_CMD)
/*----------------------------------------------------------------------------*/
/* @brief 	: write TX payload.
 	@param	: pbuf,payload data buffer
 	@param	: len,write payload data length 	
 	@retval  : 																						*/
/*----------------------------------------------------------------------------*/
#define  BC5602WriteTxPayload(pbuf,len)		BC5602WriteMultibyteRegister(WRITE_ACKPLD_CMD,pbuf,len)
/*----------------------------------------------------------------------------*/
/* @brief 	: write TX NACK payload.
 	@param	: pbuf,payload data buffer
 	@param	: len,write payload data length 	
 	@retval  : 																						*/
/*----------------------------------------------------------------------------*/
#define  BC5602WriteTxNAckPayload(pbuf,len)	BC5602WriteMultibyteRegister(WRITE_NACKPLD_CMD,pbuf,len)
/*----------------------------------------------------------------------------*/
/* @brief 	: write RX ACK payload.
 	@param	: pipe, write RX pipe
 	@param	: pbuf,payload data buffer
 	@param	: len,write payload data length 	
 	@retval  : 																						*/
/*----------------------------------------------------------------------------*/
#define  BC5602WriteRxAckPayload(pipe,pbuf,len)	BC5602WriteMultibyteRegister(WRITE_P0ACKPLD_CMD+pipe,pbuf,len)
/*----------------------------------------------------------------------------*/
/* @brief 	: read RX payload.
 	@param	: pbuf,payload data buffer
 	@param	: len,read payload data length 	
 	@retval  : 																						*/
/*----------------------------------------------------------------------------*/
#define  BC5602ReadRxPayload(pbuf,len)			BC5602ReadMultibyteRegister(READ_PLD_CMD,pbuf,len)
/*----------------------------------------------------------------------------*/
/* @brief 	: get POWER ON status.
 	@param	: 
 	@retval  :	FALSE or TRUE																	*/
/*----------------------------------------------------------------------------*/
u8 BC5602GetPWRONStatus(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: get IRQ line status.
 	@param	: 
 	@retval  :	FALSE or TRUE																	*/
/*----------------------------------------------------------------------------*/
u8	BC5602GetIRQLineStatus(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: BC5602 register set to default value
 	@param	: 
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602RegisterConfigure(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: Set crystal type.
 	@param	: xo_il - crystal oscillator low current mode.
				  xo_trim - Trim value for the internal capacitor load for the crystal.
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602CrystalSetup(u8 xo_il,u8 xo_trim);
/*----------------------------------------------------------------------------*/
/* @brief 	: wait crystal ready
 	@param	:
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602WaitCrystalReady(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: check VCO Calibration
 	@param	:
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602AutoCalibration(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: Set radio's operation mode.
 * Use this function to enter PTX (primary TX) or PRX (primary RX).
 	@param	: op_mode - operation mode select
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602SetPrimaryMode(u8 op_mode);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get operation mode
 * Use this function to get the operation mode.
 * @param	:
 * @return 	:	operation mode																	*/
/*----------------------------------------------------------------------------*/
u8		BC5602GetOperationMode(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set the CRC mode used by the radio.
 * Use this function to set the CRC mode; CRC disabled, 1 or 2 bytes.
 	@param	: 	crc_mode - CRC mode to use
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602SetCRCMode(u8 crc_mode);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Enable or disable interrupt for radio.
 * Use this function to enable or disable one of the interrupt sources for the radio.
 	@param	: 	int_source - Radio interrupt Source
					irq_state - Enable or Disable.
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602SetIRQMode(u8 int_source, u8 irq_state);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Read then clears all interrupt flags.
 * Use this function to get the interrupt flags and clear them in the same operation.
 	@param	: 	
 	@retval  :	0x10 Max Retransmit interrupt(MAXRT)									
 	@retval  :	0x20 TX Data sent interrupt(TXDS)									
 	@retval  :	0x40 RX Data received interrupt(RXDR)									*/
/*----------------------------------------------------------------------------*/
u8 	BC5602GetClearIRQFlags(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: Set radio's address width.
 * Use this function to define the radio's address width,referes to both RX and TX.
 	@param	: aw - Address with in bytes
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602SetAddressWidth(u8 aw);
/*----------------------------------------------------------------------------*/
/* @brief 	: Gets the radio's address width.
 	@param	: 
 	@retval  :	Address width																	*/
/*----------------------------------------------------------------------------*/
u8 	BC5602GetAddressWidth(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set radio's RX address and TX address.
 * Use this function to set a RX address, or to set the TX address.
 * Beware of the difference for single and multibyte address registers.
 	@param	: 	adr - Which address to set
					adrbuf - Buffer from which the address is stored in
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602SetAddress(const u8 adr, const u8 *adrbuf);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get address for selected pipe.
 * Use this function to get address for selected pipe.
 	@param	: 	adr - Which address to get
					adrbuf - buffer in which address bytes are written.
 	@retval  :	 																					*/
/*----------------------------------------------------------------------------*/
void 	BC5602GetAddress(const u8 adr,u8 *adrbuf);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set radio's on-air datarate.
 * Use this function to select radio's on-air
 	@param	: 	datarate - On-air datarate
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602SetDataRate(u8 datarate);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Open radio pipe(s) and enable/ disable auto acknowledge
 * Use this function to open one or all pipes,with or without auto acknowledge.
 	@param	: 	pipe_num - Radio pipe to open
					auto_ack - Auto_Ack ON/OFF
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602OpenPipe(u8 pipe_num, u8 auto_ack);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Close radio pipe(s).
 * Use this function to close one pipe or all pipes.
 	@param	: 	pipe_num - Pipe# number to close
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602ClosePipe(u8 pipe_num);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set auto acknowledge parameters.
 * Use this function to set retransmit and retransmit delay parameters.
 	@param	: 	retr - Number of retransmit, 0 equ retransmit OFF
 	@param	: 	delay - Retransmit delay in us Must be a 
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602SetAutoRetr(u8 retr, u8 delay);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set radio's RF channel.
 * Use this function to select which RF channel to use.
 	@param	: 	channel - RF channel
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
#define 	BC5602SetRFChannel(channel)	BC5602WriteRegister(RF_CH_REGS,channel & 0x7F)
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set radio's TX output power.
 * Use this function set the radio's TX output power.
 	@param	: 	power - Radio's TX output power
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void 	BC5602SetOutputPower(u8 power);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Read all interrupt flags.
 * Use this function to get the interrupt flags. This function is similar
 	@param	: 	
 	@retval  :	0x10 Max Retransmit interrupt(MAXRT)									
 	@retval  :	0x20 TX Data sent interrupt(TXDS)									
 	@retval  :	0x40 RX Data received interrupt(RXDR)									*/
/*----------------------------------------------------------------------------*/
#define	BC5602GetIRQFlags(void)	(u8)(BC5602ReadRegister(IRQ_STS_REGS) & (BIT6|BIT5|BIT4))
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Clear one selected interrupt flag.
 * Use this function to clear one or all spesific interrupt flag.
 	@param	: 	source - Interrupt source of which flag to clear
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
#define	BC5602ClearIRQFlags(source)		BC5602WriteRegister(IRQ_STS_REGS,source & (BIT6|BIT5|BIT4))
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set payload width for selected pipe.
 * Use this function to set the number of bytes expected on a selected pipe.
 	@param	: 	pipe_num - Pipe number to set payload width
					width - number of bytes expected
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
#define	BC5602SetRxPayloadWidth(pipe_num,width)	BC5602WriteRegister(RX_DPWP0_REGS+pipe_num,width)
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get RX payload width for selected pipe.
 * Use this function to get the expected payload width for selected ppe number.
 	@param	: 	pipe_num - Pipe number to get payload width for
 	@retval  :	Payload_Width in bytes														*/
/*----------------------------------------------------------------------------*/
#define	BC5602GetRxPayloadWidth(pipe_num)	BC5602ReadRegister(RX_DPWP0_REGS+pipe_num)	
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get auto retransmit parameters.
 * Use this function to get the auto retransmit parameters,retrans count and retrans delay.
 * @param	:
 * @return 	:	AutoRetrans Parameters
 * @retval  : 	UpperNibble Retransmit Delay
 * @retval 	:	LowerNibble Retransmit Count												*/
/*----------------------------------------------------------------------------*/
#define	BC5602GetAutoRetrStatus(void)	BC5602ReadRegister(OBSERVE_TX_REGS)
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get packet lost counter
 * Use this function to get the packet(s) counter.
 * @param	:
 * @return 	:	packet lost counter															*/
/*----------------------------------------------------------------------------*/
#define	BC5602GetPacketLostCtr(void)	(u8)(BC5602ReadRegister(OBSERVE_TX_REGS) >> 4)
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get radio's FIFO status.
 * Use this function to get the radio's FIFO status.
 	@param	: 	
 	@retval  :	return TX FIFO & RX FIFO status											*/
/*----------------------------------------------------------------------------*/
#define	BC5602GetFIFOStatus(void)	BC5602ReadRegister(FIFO_STS_REGS)
/*----------------------------------------------------------------------------*/
/* @brief 	:	Function for enabling dynamic packet length & ack payload & dynamic ack
	@param 	: 	setup - byte value enable or disable
 	@retval  :																						*/		
/*----------------------------------------------------------------------------*/
#define	BC5602SetupFeature(setup)	BC5602WriteRegister(FEATURE_CTL_REGS,setup & (BIT7 | BIT2 | BIT1 | BIT0))
/*----------------------------------------------------------------------------*/
/* @brief 	:	Function for enabling dynmic payload size.
 * @param 	:	setup - Byte value telling for which pips(s) to enable dynamic payload size
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
#define	BC5602SetupDynamicPayload(setup)	BC5602WriteRegister(DYNPL_CTL_REGS,setup & ~(BIT7 | BIT6))
/*----------------------------------------------------------------------------*/
/* @brief 	:	Reads the payload width of the received payload
 	@param	: 	
 * @retval 	:	Payload width of the received payload 									*/
/*----------------------------------------------------------------------------*/
#define	BC5602ReadRxPayloadWidth(void)	BC5602ReadRegister(RX_PLWID_REGS)
/*----------------------------------------------------------------------------*/
/* @brief 	:	Function for set CE status
	@param 	: 	setup - CE high or LOW
 	@retval  :																						*/		
/*----------------------------------------------------------------------------*/
#define	BC5602SetupCE(setup)	BC5602WriteRegister(CE_CTL_REGS,setup & BIT0)
/*----------------------------------------------------------------------------*/
/* @brief 	: 	trigger CE puls.
 	@param	: 	
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602TriggerCE(void);
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get radio's Address OK RSSI value.
 * Use this function to get the Address OK RSSI value.
 	@param	: 	
 	@retval  :	return RSSI value																*/
/*----------------------------------------------------------------------------*/
#define	BC5602GetAddressRSSI(void)		BC5602ReadRegister(RSSI_VALUE_ID_REGS)
#define	BC5602GetMeasurementRSSI(void)	BC5602ReadRegister(RSSI_VALUE_REGS)	

#endif   /* _BC5602_H_ */
